import { access, readFile } from 'node:fs/promises';
import { resolve, normalize, extname } from 'node:path';
import { createJiti } from 'jiti';
import { getTsConfigAliases } from '../config.js';
import { JsonParser, JsonObjectNode } from '@croct/json5-parser';

/**
 * Resolve an output template (string or function) into an actual path string.
 *
 * - If `outputTemplate` is a function, call it with (language, namespace)
 * - If it's a string, replace placeholders:
 *    - {{language}} or {{lng}} -> language
 *    - {{namespace}} -> namespace (or removed if namespace is undefined)
 * - Normalizes duplicate slashes and returns a platform-correct path.
 */
function getOutputPath(outputTemplate, language, namespace) {
    if (!outputTemplate) {
        // Fallback to a sensible default
        return normalize(`locales/${language}/${namespace ?? 'translation'}.json`);
    }
    if (typeof outputTemplate === 'function') {
        try {
            const result = String(outputTemplate(language, namespace));
            return normalize(result.replace(/\/\/+/g, '/'));
        }
        catch {
            // If user function throws, fallback to default path
            return normalize(`locales/${language}/${namespace ?? 'translation'}.json`);
        }
    }
    // It's a string template
    let out = String(outputTemplate);
    out = out.replace(/\{\{language\}\}|\{\{lng\}\}/g, language);
    if (namespace !== undefined && namespace !== null) {
        out = out.replace(/\{\{namespace\}\}/g, namespace);
    }
    else {
        // remove any occurrences of /{{namespace}} or {{namespace}} (keeping surrounding slashes tidy)
        out = out.replace(/\/?\{\{namespace\}\}/g, '');
    }
    // collapse duplicate slashes and normalize to platform-specific separators
    out = out.replace(/\/\/+/g, '/');
    return normalize(out);
}
/**
 * Dynamically loads a translation file, supporting .json, .js, and .ts formats.
 * @param filePath - The path to the translation file.
 * @returns The parsed content of the file, or null if not found or failed to parse.
 */
async function loadTranslationFile(filePath) {
    const fullPath = resolve(process.cwd(), filePath);
    try {
        await access(fullPath);
    }
    catch {
        return null; // File doesn't exist
    }
    try {
        const ext = extname(fullPath).toLowerCase();
        if (ext === '.json5') {
            const content = await readFile(fullPath, 'utf-8');
            // Parse as a JSON5 object node
            const node = JsonParser.parse(content, JsonObjectNode);
            return node.toJSON();
        }
        else if (ext === '.json') {
            const content = await readFile(fullPath, 'utf-8');
            return JSON.parse(content);
        }
        else if (ext === '.ts' || ext === '.js') {
            // Load TypeScript path aliases for proper module resolution
            const aliases = await getTsConfigAliases();
            const jiti = createJiti(process.cwd(), {
                alias: aliases,
                interopDefault: true,
            });
            const module = await jiti.import(fullPath, { default: true });
            return module;
        }
        return null; // Unsupported file type
    }
    catch (error) {
        console.warn(`Could not parse translation file ${filePath}:`, error);
        return null;
    }
}
// Helper to load raw JSON5 content for preservation
async function loadRawJson5Content(filePath) {
    const fullPath = resolve(process.cwd(), filePath);
    try {
        await access(fullPath);
        return await readFile(fullPath, 'utf-8');
    }
    catch {
        return null;
    }
}
/**
 * Serializes a translation object into a string based on the desired format.
 * For JSON5, preserves comments and formatting using JsonObjectNode.update().
 */
function serializeTranslationFile(data, format = 'json', indentation = 2, rawContent // Pass raw content for JSON5 preservation
) {
    const jsonString = JSON.stringify(data, null, indentation);
    switch (format) {
        case 'json5': {
            if (rawContent) {
                // Parse the original JSON5 file, update it, and output as string
                const node = JsonParser.parse(rawContent, JsonObjectNode);
                node.update(data);
                return node.toString({ object: { indentationSize: Number(indentation) ?? 2 } });
            }
            // Fallback: create a new node by parsing the generated JSON string and output as string
            const node = JsonParser.parse(jsonString, JsonObjectNode);
            return node.toString({ object: { indentationSize: Number(indentation) ?? 2 } });
        }
        case 'js':
        case 'js-esm':
            return `export default ${jsonString};\n`;
        case 'js-cjs':
            return `module.exports = ${jsonString};\n`;
        case 'ts':
            // Using `as const` provides better type inference for TypeScript users
            return `export default ${jsonString} as const;\n`;
        case 'json':
        default:
            return `${jsonString}\n`;
    }
}

export { getOutputPath, loadRawJson5Content, loadTranslationFile, serializeTranslationFile };
